var _ = require("lodash")
var errorMessage = require("../utils/errorMessage")
var clientRepository = require("../repository/clientRepository")
var agreementType = require("../utils/orderAgreementType")
var ObjectId = require("mongodb").ObjectID;
var mime = require("mime-types")
var fs = require("fs")
var utils = require("../utils/utils")

async function getClientConfig(req, res) {
    return req.httpContext.client.config
}

async function updateClientConfig(req, res) {

    var body = req.body

    if (body.processingCharge !== null && body.processingCharge !== undefined && typeof body.processingCharge != "number") {
        throw Error(errorMessage.INVALID_PROCESSING_CHARGE)
    }

    if (body.monthlyCharge !== null && body.monthlyCharge !== undefined && typeof body.monthlyCharge != "number") {
        throw Error(errorMessage.INVALID_MONTHLY_CHARGE)
    }

    if (body.overdueCharge !== null && body.overdueCharge !== undefined && typeof body.overdueCharge != "number") {
        throw Error(errorMessage.INVALID_OVERDUE_CHARGE)
    }

    if (body.monthlyChargeCount !== null && body.monthlyChargeCount !== undefined && typeof body.monthlyChargeCount != "number") {
        throw Error(errorMessage.INVALID_MONTHLY_CHARGE_COUNT)
    }

    if (body.gracePeriod !== null && body.gracePeriod !== undefined && typeof body.gracePeriod != "number") {
        throw Error(errorMessage.INVALID_GRACE_PERIOD)
    }

    if (body.creditPeriod !== null && body.creditPeriod !== undefined && typeof body.creditPeriod != "number") {
        throw Error(errorMessage.INVALID_CREDIT_PERIOD)
    }

    if (body.overduePeriod !== null && body.overduePeriod !== undefined && typeof body.overduePeriod != "number") {
        throw Error(errorMessage.INVALID_OVERDUE_PERIOD)
    }

    var result = await clientRepository.updateClientConfig(req.body, req.httpContext)
    return result
}

async function getAgreementTemplate(req, res) {
    var returnTypes = {
        url: "url",
        html: "html",
    }

    var type = req.query.type
    var returnType = returnTypes[req.query.returnType] || returnTypes.url

    if (_.isEmpty(type))
        throw Error(errorMessage.TYPE_REQUIRED)
    if (_.isEmpty(agreementType[type]))
        throw Error(errorMessage.INVALID_TYPE)

    var agreementUrl
    var defaultUrl
    var defaultPath

    switch (type) {
        case agreementType.BUYER:
            agreementUrl = req.httpContext.client.config.buyerAgreement
            defaultUrl = req.protocol + '://' + req.headers.host + '/templates/buyerAgreement.html'
            defaultPath = "template/buyerAgreement.html"
            break
        case agreementType.SELLER:
            agreementUrl = req.httpContext.client.config.sellerAgreement
            defaultUrl = req.protocol + '://' + req.headers.host + '/templates/sellerAgreement.html'
            defaultPath = "template/sellerAgreement.html"
            break
    }

    if (!_.isEmpty(agreementUrl)) {
        var filename = agreementUrl.substring(agreementUrl.lastIndexOf('/') + 1)
        var filePath = "uploads/templates/" + filename

        if (fs.existsSync(filePath)) {
            if (returnType == returnTypes.url) {
                return { url: agreementUrl }
            }
            else {
                return fs.readFileSync(filePath, "utf8")
            }
        }
        else {
            if (returnType == returnTypes.url) {
                return { url: defaultUrl }
            }
            else {
                return fs.readFileSync(defaultPath, "utf8")
            }
        }
    }
    else {
        if (returnType == returnTypes.url) {
            return { url: defaultUrl }
        }
        else {
            return fs.readFileSync(defaultPath, "utf8")
        }
    }
}

async function uploadAgreementTemplate(req, res) {

    var type = req.body.type

    if (_.isEmpty(type))
        throw Error(errorMessage.TYPE_REQUIRED)
    if (_.isEmpty(agreementType[type]))
        throw Error(errorMessage.INVALID_TYPE)

    var fileExtension

    if (req.files != undefined) {
        var file = req.files.file

        if (file === undefined)
            throw Error(errorMessage.FILE_REQUIRED)

        fileExtension = mime.extension(file.mimetype)
    }
    else {
        var { base64, mimeType } = req.body

        if (_.isEmpty(base64))
            throw Error(errorMessage.BASE64_IS_REQUIRED)
        if (_.isEmpty(mimeType))
            throw Error(errorMessage.MIMETYPE_IS_REQUIRED)

        fileExtension = mime.extension(mimeType)

        if (!fileExtension)
            throw Error(errorMessage.INVALID_MIMETYPE)
    }

    if (fileExtension != "html")
        throw Error(errorMessage.MIME_TYPE_MUST_HTML)

    var fileId = new ObjectId()
    var filename = "uploads/templates/" + fileId.toString() + "." + fileExtension
    utils.ensureDirectoryExistence(filename)

    if (file === undefined) {
        fs.writeFileSync(filename, base64, 'base64')
    }
    else {
        await file.mv(filename)
    }

    var url = req.protocol + '://' + req.headers.host + '/' + filename

    var configPatch = {}

    switch (type) {
        case agreementType.BUYER:
            configPatch.buyerAgreement = url
            break
        case agreementType.SELLER:
            configPatch.sellerAgreement = url
            break
    }

    var result = await clientRepository.updateClientConfig(configPatch, req.httpContext)
    return result
}

module.exports = {
    getClientConfig,
    updateClientConfig,
    getAgreementTemplate,
    uploadAgreementTemplate,
}