var config = require('./../config');
var _ = require('lodash');
var moment = require("moment")
var { round } = require("mathjs")
var mime = require("mime-types")
var fs = require("fs")
var uuid = require("uuid")
var utils = require("../utils/utils")
var creditOrderRepository = require('../repository/creditOrderRepository');
var creditUserRepository = require("../repository/creditUserRepository")
var transactionRepository = require("../repository/creditOrderTransactionRepository")
var fxEmailService = require("../service/fxEmailService")
var eventLogRepository = require("../repository/eventLogRepository")
var CreditOrder = require("../domain/creditOrder");
var CreditOrderTransaction = require("../domain/creditOrderTransaction")
var DeliveryOrder = require("../domain/deliveryOrder")
var ObjectId = require("mongodb").ObjectID;

var orderStatus = require("../utils/creditOrderStatus")
var orderInvoiceStatus = require("../utils/creditOrderInvoiceStatus")
var userStatus = require("../utils/creditUserStatus")
var agreementType = require("../utils/orderAgreementType")
var errorMessage = require("../utils/errorMessage")
var transactionType = require("../utils/transactionType")
var transactionMovement = require("../utils/transactionMovement")
var logType = require("../utils/eventLogType")
var actionType = require("../utils/eventLogActionType")
var deliveryOrderStatus = require("../utils/deliveryOrderStatus")

var generatePagination = require('../model/pagination')

async function getCreditOrderList(req, res) {

    var {
        limit,
        offset,
        sort,
        sortBy,
        orderId,
        orderIds,
        customerId,
        customerIds,
        userId,
        sellerId,
        sellerIds,
        status,
        statuses,
        invoiceStatus,
        invoiceStatuses,
        referenceNumber,
        toInvoiceAmount,
        fromInvoiceAmount,
        toOutstandingAmount,
        fromOutstandingAmount,
        toCreatedDate,
        fromCreatedDate,
        toAgingDate,
        fromAgingDate,
        toDeliverDate,
        fromDeliverDate,
    } = req.query

    limit = parseInt(limit || config.setting.limit)
    offset = parseInt(offset || 0)

    var result = await creditOrderRepository.getAllOrders(
        limit,
        offset,
        sort || 'asc',
        sortBy || 'id',
        orderId,
        orderIds,
        customerId,
        customerIds,
        userId,
        sellerId,
        sellerIds,
        status,
        statuses,
        invoiceStatus,
        invoiceStatuses,
        referenceNumber,
        parseFloat(toInvoiceAmount),
        parseFloat(fromInvoiceAmount),
        parseFloat(toOutstandingAmount),
        parseFloat(fromOutstandingAmount),
        toCreatedDate,
        fromCreatedDate,
        toAgingDate,
        fromAgingDate,
        toDeliverDate,
        fromDeliverDate,
        req.httpContext,
    );

    var count = await creditOrderRepository.getOrdersCount(
        orderId,
        orderIds,
        customerId,
        customerIds,
        userId,
        sellerId,
        sellerIds,
        status,
        statuses,
        invoiceStatus,
        invoiceStatuses,
        referenceNumber,
        parseFloat(toInvoiceAmount),
        parseFloat(fromInvoiceAmount),
        parseFloat(toOutstandingAmount),
        parseFloat(fromOutstandingAmount),
        toCreatedDate,
        fromCreatedDate,
        toAgingDate,
        fromAgingDate,
        toDeliverDate,
        fromDeliverDate,
        req.httpContext,
    )

    var pagination = generatePagination(offset, limit, count)

    var response = {
        result: result,
        pagination: pagination
    }

    return response
}

async function getOrdersCount(req, res) {
    var {
        orderId,
        orderIds,
        customerId,
        customerIds,
        userId,
        sellerId,
        sellerIds,
        status,
        statuses,
        invoiceStatus,
        invoiceStatuses,
        referenceNumber,
        toInvoiceAmount,
        fromInvoiceAmount,
        toOutstandingAmount,
        fromOutstandingAmount,
        toCreatedDate,
        fromCreatedDate,
        toAgingDate,
        fromAgingDate,
        toDeliverDate,
        fromDeliverDate,
    } = req.query

    var count = await creditOrderRepository.getOrdersCount(
        orderId,
        orderIds,
        customerId,
        customerIds,
        userId,
        sellerId,
        sellerIds,
        status,
        statuses,
        invoiceStatus,
        invoiceStatuses,
        referenceNumber,
        parseFloat(toInvoiceAmount),
        parseFloat(fromInvoiceAmount),
        parseFloat(toOutstandingAmount),
        parseFloat(fromOutstandingAmount),
        toCreatedDate,
        fromCreatedDate,
        toAgingDate,
        fromAgingDate,
        toDeliverDate,
        fromDeliverDate,
        req.httpContext,
    )

    return { totalItem: count }
}

async function getOrderHistory(req, res) {
    var id = req.params.orderId

    var {
        limit,
        offset,
        sort,
        sortBy,
    } = req.query

    var isExist = await creditOrderRepository.isOrderExist(id, req.httpContext)

    if (!isExist)
        throw Error(errorMessage.ORDER_NOT_FOUND)

    limit = parseInt(limit || config.setting.limit)
    offset = parseInt(offset || 0)

    var result = await eventLogRepository.getAllLogs(
        limit,
        offset,
        sort || 'asc',
        sortBy || 'id',
        null,
        null,
        null,
        [logType.ORDER, logType.INVOICE, logType.DELIVERY_ORDER],
        id,
        req.httpContext
    )

    var count = await eventLogRepository.getLogCount(
        null,
        null,
        null,
        [logType.ORDER, logType.INVOICE, logType.DELIVERY_ORDER],
        id,
        req.httpContext
    )

    var pagination = generatePagination(offset, limit, count)

    var response = {
        result: result,
        pagination: pagination
    }

    return response
}

async function getCreditOrder(req, res) {
    var id = req.params.orderId;

    return await creditOrderRepository.getOrder(id, req.httpContext,)
}

async function createCreditOrder(req, res) {
    var body = req.body;

    var creditOrder = new CreditOrder(body);
    creditOrder.status = orderStatus.PENDING

    creditOrder.invoiceUrl = null
    creditOrder.invoiceStatus = null
    creditOrder.deliveryOrders = []
    creditOrder.agingDatetime = null
    creditOrder.deliverDatetime = null

    if (_.isEmpty(creditOrder.orderId))
        throw Error(errorMessage.ORDER_ID_REQUIRED)

    if (_.isEmpty(creditOrder.buyer.customerId))
        throw Error(errorMessage.CUSTOMER_ID_REQUIRED)

    if (_.isEmpty(creditOrder.seller.sellerId))
        throw Error(errorMessage.SELLER_ID_REQUIRED)

    if (creditOrder.invoiceAmount === null || creditOrder.invoiceAmount === undefined)
        throw Error(errorMessage.INVOICE_AMOUNT_REQUIRED)

    if (typeof creditOrder.invoiceAmount != "number")
        throw Error(errorMessage.INVALID_INVOICE_AMOUNT)

    var isExist = await creditOrderRepository.isOrderExist(creditOrder.orderId, req.httpContext,)

    if (isExist)
        throw Error(errorMessage.ORDER_ALREADY_EXIST)

    var user = await creditUserRepository.getUser(creditOrder.buyer.customerId, req.httpContext,)
    creditOrder.buyer.userId = user.id

    if (user.status != userStatus.ENABLED)
        throw Error(errorMessage.CREDIT_DISABLED)

    var outstandingAmount

    if (creditOrder.outstandingAmount != null && creditOrder.outstandingAmount != undefined && typeof creditOrder.outstandingAmount == "number") {
        outstandingAmount = creditOrder.outstandingAmount
    }
    else {
        outstandingAmount = creditOrder.invoiceAmount + (creditOrder.invoiceAmount * req.httpContext.client.config.processingCharge)
            + (req.httpContext.client.config.monthlyChargeCount * creditOrder.invoiceAmount * req.httpContext.client.config.monthlyCharge)
    }
    outstandingAmount = round(outstandingAmount, config.setting.decimal)

    var updatedCreditUsed = user.creditUsed + outstandingAmount
    updatedCreditUsed = round(updatedCreditUsed, config.setting.decimal)

    var updatedTotalCreditUsed = user.totalCreditUsed + outstandingAmount
    updatedTotalCreditUsed = round(updatedTotalCreditUsed, config.setting.decimal)

    if (updatedCreditUsed > user.creditLimit)
        throw Error(errorMessage.INSUFFICIENT_CREDIT)

    creditOrder.outstandingAmount = outstandingAmount
    creditOrder.totalOutstandingAmount = outstandingAmount
    user.creditUsed = updatedCreditUsed
    user.totalCreditUsed = updatedTotalCreditUsed

    var agreementData = {
        buyerName: creditOrder.buyer.name,
        buyerAddress: creditOrder.buyer.deliveryAddress,
        buyerPhone: creditOrder.buyer.phoneNo,
        sellerName: creditOrder.seller.name,
        sellerShopName: creditOrder.seller.shopName,
        sellerAddress: creditOrder.seller.address,
        orderReference: user.creditOrderReference,
        orderPlacedDate: moment().format("yyyy-MM-DD"),
        orderPlacedTimestamp: moment().format("yyyy-MM-DD hh:mm a"),
    }
    var fileId = new ObjectId()
    var buyerHtml = await utils.generateAgreement(agreementType.BUYER, agreementData, req.httpContext)
    var buyerFilename = "uploads/agreements/" + fileId.toString() + ".html"
    utils.ensureDirectoryExistence(buyerFilename)
    fs.writeFileSync(buyerFilename, buyerHtml)
    creditOrder.buyer.agreementUrl = req.protocol + '://' + req.headers.host + '/uploads/agreements/' + fileId.toString() + '.html'

    creditOrder.referenceNumber = user.creditOrderReference

    user.creditOrderReference = uuid.v4().substr(24, 12).toUpperCase()

    await creditUserRepository.updateUser(user.customerId, user, req.httpContext,)

    var result = await creditOrderRepository.createOrder(creditOrder, req.httpContext,);

    await utils.logEvent(req, actionType.CREATE, logType.ORDER, null, creditOrder, creditOrder.orderId)

    return result
}

async function calculateInvoiceAmount(req, res) {

    var { invoiceAmount } = req.body

    if (!utils.isValidNumber(invoiceAmount))
        throw Error(errorMessage.INVALID_INVOICE_AMOUNT)

    var outstandingAmount = invoiceAmount + (invoiceAmount * req.httpContext.client.config.processingCharge)
        + (req.httpContext.client.config.monthlyChargeCount * invoiceAmount * req.httpContext.client.config.monthlyCharge)
    outstandingAmount = round(outstandingAmount, config.setting.decimal)

    var processingFee = (invoiceAmount * req.httpContext.client.config.processingCharge)
        + (req.httpContext.client.config.monthlyChargeCount * invoiceAmount * req.httpContext.client.config.monthlyCharge)
    processingFee = round(processingFee, config.setting.decimal)

    var monthlyFee = invoiceAmount * req.httpContext.client.config.monthlyCharge
    monthlyFee = round(monthlyFee, config.setting.decimal)

    return { total: outstandingAmount, processingFee: processingFee, monthlyFee: monthlyFee }
}

async function updateInvoiceAmount(req, res) {
    var id = req.params.orderId

    var { invoiceAmount, outstandingAmount } = req.body

    var creditOrder = await creditOrderRepository.getOrder(id, req.httpContext)

    if (creditOrder.status != orderStatus.PENDING)
        throw Error(errorMessage.CANNOT_UPDATE_INVOICE_AMOUNT)

    if (!utils.isValidNumber(invoiceAmount))
        throw Error(errorMessage.INVALID_INVOICE_AMOUNT)

    var user = await creditUserRepository.getUser(creditOrder.buyer.customerId, req.httpContext)

    if (outstandingAmount == null || outstandingAmount == undefined || typeof outstandingAmount != "number") {
        outstandingAmount = invoiceAmount + (invoiceAmount * req.httpContext.client.config.processingCharge)
            + (req.httpContext.client.config.monthlyChargeCount * invoiceAmount * req.httpContext.client.config.monthlyCharge)
    }
    outstandingAmount = round(outstandingAmount, config.setting.decimal)

    var diff = outstandingAmount - creditOrder.outstandingAmount

    var updatedCreditUsed = user.creditUsed + diff
    updatedCreditUsed = round(updatedCreditUsed, config.setting.decimal)

    var updatedTotalCreditUsed = user.totalCreditUsed + diff
    updatedTotalCreditUsed = round(updatedTotalCreditUsed, config.setting.decimal)

    if (updatedCreditUsed > user.creditLimit)
        throw Error(errorMessage.INSUFFICIENT_CREDIT)

    var updateUser = { creditUsed: updatedCreditUsed, totalCreditUsed: updatedTotalCreditUsed }

    await creditUserRepository.updateUser(user.customerId, updateUser, req.httpContext)

    var updateOrder = { invoiceAmount: invoiceAmount, outstandingAmount: outstandingAmount, totalOutstandingAmount: outstandingAmount }

    var result = await creditOrderRepository.updateOrder(id, updateOrder, req.httpContext)

    await utils.logEvent(req, actionType.UPDATE, logType.ORDER, creditOrder, updateOrder, id)

    return result
}

async function updateCreditOrder(req, res) {
    var id = req.params.orderId

    var body = req.body;

    var creditOrder = await creditOrderRepository.getOrder(id, req.httpContext,)

    if (creditOrder.status == orderStatus.CLOSED ||
        creditOrder.status == orderStatus.CANCELLED ||
        creditOrder.status == orderStatus.REJECTED)
        throw Error(errorMessage.CANNOT_UPDATE_ORDER)

    var updateCreditOrder = new CreditOrder(body);
    updateCreditOrder.outstandingAmount = creditOrder.outstandingAmount

    delete updateCreditOrder.agingDatetime
    delete updateCreditOrder.deliverDatetime
    delete updateCreditOrder.totalOutstandingAmount

    if (!_.isEmpty(updateCreditOrder.invoiceStatus)) {
        if (_.isEmpty(orderInvoiceStatus[body.invoiceStatus]))
            throw Error(errorMessage.INVALID_INVOICE_STATUS)

        if (_.isEmpty(creditOrder.invoiceUrl))
            throw Error(errorMessage.CANNOT_UPDATE_INVOICE_STATUS)
    }

    if (!_.isEmpty(updateCreditOrder.status)) {
        if (_.isEmpty(orderStatus[body.status]))
            throw Error(errorMessage.INVALID_STATUS)

        if (updateCreditOrder.status == orderStatus.CLOSED && creditOrder.outstandingAmount > 0) {
            throw Error(errorMessage.CANNOT_CLOSE_ORDER)
        }
        //Delivered Pending Payment changes
        if (updateCreditOrder.status == orderStatus.DELIVERED_PENDING_PAYMENT) {
            var agingDate = new Date()
            agingDate.setDate(agingDate.getDate() + req.httpContext.client.config.creditPeriod)
            updateCreditOrder.agingDatetime = agingDate

            updateCreditOrder.deliverDatetime = new Date()
        }

        if (updateCreditOrder.status == orderStatus.CANCELLED ||
            updateCreditOrder.status == orderStatus.REJECTED) {
            var transaction = new CreditOrderTransaction({
                orderId: creditOrder.orderId,
                changeAmount: creditOrder.outstandingAmount,
                sign: transactionMovement.DEDUCTION,
                type: transactionType.REFUND,
                metadata: { remarks: 'ORDER ' + updateCreditOrder.status }
            })

            var user = await creditUserRepository.getUser(creditOrder.buyer.customerId, req.httpContext,)
            user.creditUsed = user.creditUsed - transaction.changeAmount

            transaction.previousAmount = creditOrder.outstandingAmount
            updateCreditOrder.outstandingAmount = 0
            transaction.currentAmount = 0

            await transactionRepository.createTransaction(transaction, req.httpContext,)
            await creditUserRepository.updateUser(creditOrder.buyer.customerId, user, req.httpContext,)
        }

        if (updateCreditOrder.status == orderStatus.CLOSED ||
            updateCreditOrder.status == orderStatus.REJECTED ||
            updateCreditOrder.status == orderStatus.CANCELLED) {
            updateCreditOrder.agingDatetime = null
        }
    }

    delete updateCreditOrder.invoiceAmount
    delete updateCreditOrder.invoiceUrl
    delete updateCreditOrder.deliveryOrders

    var result = await creditOrderRepository.updateOrder(id, updateCreditOrder, req.httpContext,)

    delete updateCreditOrder.outstandingAmount

    await utils.logEvent(req, actionType.UPDATE, logType.ORDER, creditOrder, updateCreditOrder, id)

    return result
}

async function uploadInvoice(req, res) {

    var file = req.files.file

    if (file === undefined)
        throw Error(errorMessage.FILE_REQUIRED)

    var fileExtension = mime.extension(file.mimetype)

    var orderId = req.params.orderId

    var creditOrder = await creditOrderRepository.getOrder(orderId, req.httpContext,)

    if (creditOrder.status == orderStatus.CLOSED ||
        creditOrder.status == orderStatus.CANCELLED ||
        creditOrder.status == orderStatus.REJECTED)
        throw Error(errorMessage.CANNOT_UPDATE_ORDER)

    if (!_.isEmpty(creditOrder.invoiceUrl) && creditOrder.invoiceStatus == orderInvoiceStatus.APPROVED) {
        throw Error(errorMessage.ALREADY_HAS_APPROVED_INVOICE)
    }

    var updateCreditOrder = {}

    var fileId = new ObjectId()
    var filename = "uploads/invoices/" + fileId.toString() + "." + fileExtension
    utils.ensureDirectoryExistence(filename)
    await file.mv(filename)

    updateCreditOrder.invoiceUrl = req.protocol + '://' + req.headers.host + '/uploads/invoices/' + fileId.toString() + '.' + fileExtension
    updateCreditOrder.invoiceStatus = orderInvoiceStatus.PENDING

    var result = await creditOrderRepository.updateOrder(orderId, updateCreditOrder, req.httpContext,)

    await utils.logEvent(req, actionType.UPLOAD, logType.INVOICE, creditOrder, updateCreditOrder, orderId)

    return result
}

async function uploadInvoiceBase64(req, res) {
    var { base64, mimeType } = req.body

    if (_.isEmpty(base64))
        throw Error(errorMessage.BASE64_IS_REQUIRED)
    if (_.isEmpty(mimeType))
        throw Error(errorMessage.MIMETYPE_IS_REQUIRED)

    var fileExtension = mime.extension(mimeType)

    if (!fileExtension)
        throw Error(errorMessage.INVALID_MIMETYPE)

    var orderId = req.params.orderId

    var creditOrder = await creditOrderRepository.getOrder(orderId, req.httpContext,)

    if (creditOrder.status == orderStatus.CLOSED ||
        creditOrder.status == orderStatus.CANCELLED ||
        creditOrder.status == orderStatus.REJECTED)
        throw Error(errorMessage.CANNOT_UPDATE_ORDER)

    if (!_.isEmpty(creditOrder.invoiceUrl) && creditOrder.invoiceStatus == orderInvoiceStatus.APPROVED) {
        throw Error(errorMessage.ALREADY_HAS_APPROVED_INVOICE)
    }

    var updateCreditOrder = {}

    var fileId = new ObjectId()
    var filename = "uploads/invoices/" + fileId.toString() + "." + fileExtension
    utils.ensureDirectoryExistence(filename)
    fs.writeFileSync(filename, base64, 'base64')

    updateCreditOrder.invoiceUrl = req.protocol + '://' + req.headers.host + '/uploads/invoices/' + fileId.toString() + '.' + fileExtension
    updateCreditOrder.invoiceStatus = orderInvoiceStatus.PENDING

    var result = await creditOrderRepository.updateOrder(orderId, updateCreditOrder, req.httpContext,)

    await utils.logEvent(req, actionType.UPLOAD, logType.INVOICE, creditOrder, updateCreditOrder, orderId)

    return result
}

async function deleteInvoice(req, res) {
    var orderId = req.params.orderId

    var creditOrder = await creditOrderRepository.getOrder(orderId, req.httpContext,)

    if (creditOrder.status == orderStatus.CLOSED ||
        creditOrder.status == orderStatus.CANCELLED ||
        creditOrder.status == orderStatus.REJECTED)
        throw Error(errorMessage.CANNOT_UPDATE_ORDER)

    if (creditOrder.status == orderStatus.SHIPPED)
        throw Error(errorMessage.CANNOT_DELETE_INVOICE)

    if (creditOrder.invoiceStatus == orderInvoiceStatus.APPROVED)
        throw Error(errorMessage.CANNOT_DELETE_APPROVED_INVOICE)

    if (_.isEmpty(creditOrder.invoiceUrl))
        throw Error(errorMessage.ALREADY_DELETED)

    var result = await creditOrderRepository.deleteInvoice(orderId, req.httpContext,)

    await utils.logEvent(req, actionType.DELETE, logType.INVOICE, creditOrder, { invoiceUrl: null, invoiceStatus: null }, orderId)

    return result
}

async function getDeliveryOrders(req, res) {
    var {
        limit,
        offset,
        sort,
        sortBy,
        orderId,
        status,
        referenceNumber,
    } = req.query

    limit = parseInt(limit || config.setting.limit)
    offset = parseInt(offset || 0)

    var result = await creditOrderRepository.getDeliveryOrders(
        limit,
        offset,
        sort,
        sortBy,
        orderId,
        status,
        referenceNumber,
        req.httpContext,
    );

    var count = await creditOrderRepository.getDeliveryOrdersCount(
        orderId,
        status,
        referenceNumber,
        req.httpContext,
    )

    var pagination = generatePagination(offset, limit, count)

    var response = {
        result: result,
        pagination: pagination
    }

    return response
}

async function uploadDeliveryOrder(req, res) {
    var { base64, mimeType, remarks } = req.body

    if (_.isEmpty(base64))
        throw Error(errorMessage.BASE64_IS_REQUIRED)
    if (_.isEmpty(mimeType))
        throw Error(errorMessage.MIMETYPE_IS_REQUIRED)

    var fileExtension = mime.extension(mimeType)

    if (!fileExtension)
        throw Error(errorMessage.INVALID_MIMETYPE)

    var orderId = req.params.orderId

    var creditOrder = await creditOrderRepository.getOrder(orderId, req.httpContext,)

    if (creditOrder.status == orderStatus.CLOSED ||
        creditOrder.status == orderStatus.CANCELLED ||
        creditOrder.status == orderStatus.REJECTED)
        throw Error(errorMessage.CANNOT_UPDATE_ORDER)

    var currentTime = new Date()

    var deliveryOrder = new DeliveryOrder({
        _id: new ObjectId(),
        remarks: remarks,
        status: deliveryOrderStatus.PENDING,
        createdDatetime: currentTime,
        createdBy: req.httpContext.identity,
        lastModifiedDatetime: currentTime,
        lastModifiedBy: req.httpContext.identity,
    })

    var fileId = new ObjectId()
    var filename = "uploads/deliveryOrders/" + fileId.toString() + "." + fileExtension
    utils.ensureDirectoryExistence(filename)
    fs.writeFileSync(filename, base64, 'base64')

    deliveryOrder.deliveryOrderUrl = req.protocol + '://' + req.headers.host + '/uploads/deliveryOrders/' + fileId.toString() + '.' + fileExtension

    creditOrder.deliveryOrders.push(deliveryOrder)

    // var orderUrl = req.httpContext.client.config.prestaDomain + '/index.php?controller=order-detail&id_order=' + orderId

    // await fxEmailService.sendDeliveryOrderEmail(
    //     creditOrder.buyer.customerId,
    //     creditOrder.orderId,
    //     creditOrder.buyer.name,
    //     creditOrder.seller.name,
    //     orderUrl,
    //     req.httpContext,
    // )

    var result = await creditOrderRepository.updateDeliveryOrder(orderId, creditOrder.deliveryOrders, req.httpContext,)

    await utils.logEvent(req, actionType.UPLOAD, logType.DELIVERY_ORDER, null, { deliveryOrder: deliveryOrder }, orderId)

    return result
}

async function updateDeliveryOrder(req, res) {
    var { base64, mimeType, remarks, status } = req.body

    if (_.isEmpty(base64) && _.isEmpty(mimeType) && _.isEmpty(remarks) && _.isEmpty(status))
        throw Error(errorMessage.MUST_PATCH_AT_LEAST_ONE_FIELD)

    var orderId = req.params.orderId

    var creditOrder = await creditOrderRepository.getOrder(orderId, req.httpContext,)

    if (creditOrder.status == orderStatus.CLOSED ||
        creditOrder.status == orderStatus.CANCELLED ||
        creditOrder.status == orderStatus.REJECTED)
        throw Error(errorMessage.CANNOT_UPDATE_ORDER)

    var deliveryOrderId = req.params.deliveryOrderId

    var index = creditOrder.deliveryOrders.findIndex((element, index) => {
        if (element.id === deliveryOrderId) {
            return true
        }
    })

    if (index === -1)
        throw Error(errorMessage.INVALID_DELIVERY_ORDER_ID)

    var deliveryOrder = creditOrder.deliveryOrders[index]
    var updateDeliveryOrder = utils.clone(deliveryOrder)

    if (!_.isEmpty(deliveryOrder.status) && (deliveryOrder.status == deliveryOrderStatus.APPROVED || deliveryOrder.status == deliveryOrderStatus.REJECTED)) {
        throw Error(errorMessage.CANNOT_UPDATE_DELIVERY_ORDER)
    }

    var currentTime = new Date()

    updateDeliveryOrder.lastModifiedDatetime = currentTime
    updateDeliveryOrder.lastModifiedBy = req.httpContext.identity

    if (!_.isEmpty(remarks))
        updateDeliveryOrder.remarks = remarks

    if (!_.isEmpty(status)) {
        if (_.isEmpty(deliveryOrderStatus[status]))
            throw Error(errorMessage.INVALID_STATUS)

        updateDeliveryOrder.status = status
    }


    if (!_.isEmpty(base64) || !_.isEmpty(mimeType)) {
        if (_.isEmpty(base64))
            throw Error(errorMessage.BASE64_IS_REQUIRED)
        if (_.isEmpty(mimeType))
            throw Error(errorMessage.MIMETYPE_IS_REQUIRED)

        var fileExtension = mime.extension(mimeType)

        if (!fileExtension)
            throw Error(errorMessage.INVALID_MIMETYPE)

        var fileId = new ObjectId()
        var filename = "uploads/deliveryOrders/" + fileId.toString() + "." + fileExtension
        utils.ensureDirectoryExistence(filename)
        fs.writeFileSync(filename, base64, 'base64')

        updateDeliveryOrder.deliveryOrderUrl = req.protocol + '://' + req.headers.host + '/uploads/deliveryOrders/' + fileId.toString() + '.' + fileExtension
    }

    creditOrder.deliveryOrders[index] = updateDeliveryOrder

    var result = await creditOrderRepository.updateDeliveryOrder(orderId, creditOrder.deliveryOrders, req.httpContext,)

    await utils.logEvent(req, actionType.UPDATE, logType.DELIVERY_ORDER, { deliveryOrder: deliveryOrder }, { deliveryOrder: updateDeliveryOrder }, orderId)

    return result
}

async function deleteDeliveryOrder(req, res) {
    var { orderId, deliveryOrderId } = req.params

    var creditOrder = await creditOrderRepository.getOrder(orderId, req.httpContext,)

    if (creditOrder.status == orderStatus.CLOSED ||
        creditOrder.status == orderStatus.CANCELLED ||
        creditOrder.status == orderStatus.REJECTED)
        throw Error(errorMessage.CANNOT_UPDATE_ORDER)

    var index = creditOrder.deliveryOrders.findIndex((element, index) => {
        if (element.id === deliveryOrderId) {
            return true
        }
    })

    if (index === -1)
        throw Error(errorMessage.INVALID_DELIVERY_ORDER_ID)

    var deliveryOrder = creditOrder.deliveryOrders[index]

    if (!_.isEmpty(deliveryOrder.status) && (deliveryOrder.status == deliveryOrderStatus.APPROVED || deliveryOrder.status == deliveryOrderStatus.REJECTED)) {
        throw Error(errorMessage.CANNOT_DELETE_DELIVERY_ORDER)
    }

    creditOrder.deliveryOrders.splice(index, 1)

    var result = await creditOrderRepository.updateDeliveryOrder(orderId, creditOrder.deliveryOrders, req.httpContext,)

    await utils.logEvent(req, actionType.DELETE, logType.DELIVERY_ORDER, { deliveryOrder: deliveryOrder }, null, orderId)

    return result
}

async function saveSellerAgreement(req, res) {
    var orderId = req.params.orderId

    var creditOrder = await creditOrderRepository.getOrder(orderId, req.httpContext,)

    if (!_.isEmpty(creditOrder.seller.agreementUrl))
        throw Error(errorMessage.ALREADY_HAS_AGREEMENT)

    var agreementData = {
        buyerName: creditOrder.buyer.name,
        buyerAddress: creditOrder.buyer.deliveryAddress,
        buyerPhone: creditOrder.buyer.phoneNo,
        sellerName: creditOrder.seller.name,
        sellerShopName: creditOrder.seller.shopName,
        sellerAddress: creditOrder.seller.address,
        orderReference: creditOrder.referenceNumber,
        orderPlacedDate: moment(creditOrder.createdDatetime).format("yyyy-MM-DD"),
        orderPlacedTimestamp: moment(creditOrder.createdDatetime).format("yyyy-MM-DD hh:mm a"),
        orderApprovedTimestamp: moment().format("yyyy-MM-DD hh:mm a"),
    }

    var fileId = new ObjectId()
    var sellerHtml = await utils.generateAgreement(agreementType.SELLER, agreementData, req.httpContext)
    var sellerFilename = "uploads/agreements/" + fileId.toString() + ".html"
    utils.ensureDirectoryExistence(sellerFilename)
    fs.writeFileSync(sellerFilename, sellerHtml)
    creditOrder.seller.agreementUrl = req.protocol + '://' + req.headers.host + '/uploads/agreements/' + fileId.toString() + '.html'

    fileId = new ObjectId()
    var buyerHtml = await utils.generateAgreement(agreementType.BUYER, agreementData, req.httpContext)
    var buyerFilename = "uploads/agreements/" + fileId.toString() + ".html"
    utils.ensureDirectoryExistence(buyerFilename)
    fs.writeFileSync(buyerFilename, buyerHtml)
    creditOrder.buyer.agreementUrl = req.protocol + '://' + req.headers.host + '/uploads/agreements/' + fileId.toString() + '.html'

    var result = await creditOrderRepository.updateOrder(orderId, creditOrder, req.httpContext,)

    await utils.logEvent(req, actionType.UPDATE, logType.ORDER, { seller: { agreementUrl: null } }, { seller: { agreementUrl: creditOrder.seller.agreementUrl } }, orderId)

    return result
}

module.exports = {
    getCreditOrderList,
    getOrdersCount,
    getCreditOrder,
    createCreditOrder,
    updateCreditOrder,
    uploadInvoice,
    uploadInvoiceBase64,
    deleteInvoice,
    uploadDeliveryOrder,
    updateDeliveryOrder,
    deleteDeliveryOrder,
    saveSellerAgreement,
    getOrderHistory,
    updateInvoiceAmount,
    calculateInvoiceAmount,
    getDeliveryOrders,
}