var config = require('./../config');
var utils = require("../utils/utils")
var _ = require('lodash');
var { round } = require("mathjs")

var transactionRepository = require('../repository/creditOrderTransactionRepository');
var creditOrderRepository = require("../repository/creditOrderRepository")
var creditPaymentRepository = require("../repository/creditPaymentRepository")
var creditUserRepository = require("../repository/creditUserRepository")

var CreditOrderTransaction = require("../domain/creditOrderTransaction");

var transactionType = require("../utils/transactionType")
var transactionMovement = require("../utils/transactionMovement")
var paymentStatus = require("../utils/creditPaymentStatus");
var errorMessage = require("../utils/errorMessage")
var logType = require("../utils/eventLogType")
var actionType = require("../utils/eventLogActionType")
var userStatus = require("../utils/creditUserStatus")

var generatePagination = require('../model/pagination')

async function getTransactionList(req, res) {
    var {
        limit,
        offset,
        sort,
        sortBy,
        orderId,
        sign,
        paymentId,
        type,
        toChangeAmount,
        fromChangeAmount,
    } = req.query

    limit = parseInt(limit || config.setting.limit)
    offset = parseInt(offset || 0)

    var result = await transactionRepository.getAllTransactions(
        limit,
        offset,
        sort || 'asc',
        sortBy || 'id',
        orderId,
        sign,
        paymentId,
        type,
        parseFloat(toChangeAmount),
        parseFloat(fromChangeAmount),
        req.httpContext,
    );

    var count = await transactionRepository.getTransactionsCount(
        orderId,
        sign,
        paymentId,
        type,
        parseFloat(toChangeAmount),
        parseFloat(fromChangeAmount),
        req.httpContext,
    )

    var pagination = generatePagination(offset, limit, count)

    var response = {
        result: result,
        pagination: pagination
    }

    return response
}

async function getTransaction(req, res) {
    var id = req.params.transactionId;

    return await transactionRepository.getTransaction(id, req.httpContext,)
}

async function createTransaction(req, res) {
    var body = req.body;

    var transaction = new CreditOrderTransaction(body);

    if (_.isEmpty(transaction.orderId))
        throw Error(errorMessage.ORDER_ID_REQUIRED)

    if (transaction.changeAmount === null ||
        transaction.changeAmount == undefined)
        throw Error(errorMessage.CHANGE_AMOUNT_REQUIRED)

    if (typeof transaction.changeAmount != "number")
        throw Error(errorMessage.INVALID_CHANGE_AMOUNT)

    if (_.isEmpty(transaction.type))
        throw Error(errorMessage.TYPE_REQUIRED)

    if (_.isEmpty(transactionType[transaction.type]))
        throw Error(errorMessage.INVALID_TYPE)

    if (_.isEmpty(transaction.paymentId) &&
        transaction.type === transactionType.PAYMENT)
        throw Error(errorMessage.PAYMENT_ID_REQUIRED)

    if (_.isEmpty(transaction.sign))
        throw Error(errorMessage.SIGN_REQUIRED)

    if (_.isEmpty(transactionMovement[transaction.sign]))
        throw Error(errorMessage.INVALID_SIGN)

    if (transaction.type === transactionType.REFUND &&
        transaction.sign !== transactionMovement.DEDUCTION)
        throw Error(errorMessage.INVALID_SIGN_WHEN_REFUND)


    var creditOrder = await creditOrderRepository.getOrder(transaction.orderId, req.httpContext,)

    if (transaction.type == transactionType.AGING) {
        if (_.isEmpty(creditOrder.agingDatetime) || creditOrder.agingDatetime > new Date())
            throw Error(errorMessage.ORDER_NOT_OVERDUE)
    }

    if (transaction.sign === transactionMovement.DEDUCTION &&
        transaction.changeAmount > creditOrder.outstandingAmount)
        throw Error(errorMessage.CHANGE_AMOUNT_EXCEEDED_OUTSTANDING)

    if (!_.isEmpty(transaction.paymentId)) {
        if (transaction.type != transactionType.PAYMENT || transaction.sign != transactionMovement.DEDUCTION)
            throw Error(errorMessage.INVALID_PAYMENT_TYPE_OR_SIGN)

        var payment = await creditPaymentRepository.getPayment(transaction.paymentId, req.httpContext,)
        if (payment.status != paymentStatus.APPROVED)
            throw Error(errorMessage.PAYMENT_NOT_APPROVED_YET)

        if (transaction.changeAmount > payment.contra)
            throw Error(errorMessage.CHANGE_AMOUNT_EXCEEDED_CONTRA)
            
        var updatePayment = {contra: payment.contra - transaction.changeAmount}

        await creditPaymentRepository.updatePayment(payment.id, updatePayment, req.httpContext)
    }

    var user = await creditUserRepository.getUser(creditOrder.buyer.customerId, req.httpContext,)

    var updateUser = {}
    updateUser.creditUsed = transaction.sign === transactionMovement.ADDITION ?
        user.creditUsed + transaction.changeAmount :
        user.creditUsed - transaction.changeAmount
    updateUser.creditUsed = round(updateUser.creditUsed, config.setting.decimal)

    updateUser.totalCreditUsed = transaction.sign === transactionMovement.ADDITION ? 
        user.totalCreditUsed + transaction.changeAmount : 
        user.totalCreditUsed
    updateUser.totalCreditUsed = round(updateUser.totalCreditUsed, config.setting.decimal)

    if (!_.isEmpty(transaction.paymentId)) {
        updateUser.totalContra = user.totalContra - transaction.changeAmount
        updateUser.totalContra = round(updateUser.totalContra, config.setting.decimal)
    }

    transaction.previousAmount = creditOrder.outstandingAmount

    var updatedOutstandingAmount = transaction.sign === transactionMovement.ADDITION ?
        creditOrder.outstandingAmount + transaction.changeAmount :
        creditOrder.outstandingAmount - transaction.changeAmount
    updatedOutstandingAmount = round(updatedOutstandingAmount, config.setting.decimal)

    var updatedTotalOutstandingAmount = transaction.sign === transactionMovement.ADDITION ? 
        creditOrder.totalOutstandingAmount + transaction.changeAmount :
        creditOrder.totalOutstandingAmount
    updatedTotalOutstandingAmount = round(updatedTotalOutstandingAmount, config.setting.decimal)

    var updateCreditOrder = { outstandingAmount: updatedOutstandingAmount, totalOutstandingAmount: updatedTotalOutstandingAmount }

    transaction.currentAmount = updatedOutstandingAmount

    if (transaction.type == transactionType.AGING) {
        var agingDate = new Date(creditOrder.agingDatetime)
        agingDate.setDate(agingDate.getDate() + req.httpContext.client.config.overduePeriod)
        updateCreditOrder.agingDatetime = agingDate

        updateUser.status = userStatus.BLOCKED
    }

    var result = await transactionRepository.createTransaction(transaction, req.httpContext,)

    await creditOrderRepository.updateOrder(transaction.orderId, updateCreditOrder, req.httpContext,)
    await creditUserRepository.updateUser(creditOrder.buyer.customerId, updateUser, req.httpContext,)

    await utils.logEvent(req, actionType.UPDATE, logType.USER, user, updateUser, user.customerId)
    await utils.logEvent(req, actionType.UPDATE, logType.ORDER, creditOrder, updateCreditOrder, creditOrder.orderId)
    await utils.logEvent(req, actionType.CREATE, logType.TRANSACTION, null, transaction, result.id)

    return result
}

async function updateTransaction(req, res) {
    var id = req.params.transactionId

    var body = req.body;

    var transaction = new CreditOrderTransaction(body);

    return await transactionRepository.updateTransaction(id, transaction, req.httpContext,)
}

module.exports = {
    getTransactionList,
    getTransaction,
    createTransaction,
    updateTransaction
}