var config = require('./../config');
var _ = require('lodash');
var fs = require("fs")
var mime = require("mime-types")
var utils = require("../utils/utils")
var { round } = require("mathjs")
var ObjectId = require("mongodb").ObjectID;
var creditPaymentRepository = require('../repository/creditPaymentRepository');
var creditUserRepository = require("../repository/creditUserRepository")
var eventLogRepository = require("../repository/eventLogRepository")
var fxEmailService = require("../service/fxEmailService")
var CreditPayment = require("../domain/creditPayment");

var paymentStatus = require("../utils/creditPaymentStatus")
var errorMessage = require("../utils/errorMessage")
var logType = require("../utils/eventLogType")
var actionType = require("../utils/eventLogActionType")

var generatePagination = require('../model/pagination');

async function getPaymentList(req, res) {

    var {
        limit,
        offset,
        sort,
        sortBy,
        customerId,
        customerIds,
        status,
        referenceNumber,
        toAmount,
        fromAmount,
        toContra,
        fromContra,
        toCreatedDate,
        fromCreatedDate,
    } = req.query

    limit = parseInt(limit || config.setting.limit)
    offset = parseInt(offset || 0)

    var result = await creditPaymentRepository.getAllPayments(
        limit,
        offset,
        sort || 'asc',
        sortBy || 'id',
        customerId,
        customerIds,
        status,
        referenceNumber,
        parseFloat(toAmount),
        parseFloat(fromAmount),
        parseFloat(toContra),
        parseFloat(fromContra),
        toCreatedDate,
        fromCreatedDate,
        req.httpContext,
    );

    var count = await creditPaymentRepository.getPaymentsCount(
        customerId,
        customerIds,
        status,
        referenceNumber,
        parseFloat(toAmount),
        parseFloat(fromAmount),
        parseFloat(toContra),
        parseFloat(fromContra),
        toCreatedDate,
        fromCreatedDate,
        req.httpContext,
    )

    var pagination = generatePagination(offset, limit, count)

    var response = {
        result: result,
        pagination: pagination
    }

    return response
}

async function getPaymentHistory(req, res) {
    var id = req.params.paymentId

    var {
        limit,
        offset,
        sort,
        sortBy,
    } = req.query

    await creditPaymentRepository.getPayment(id, req.httpContext)

    limit = parseInt(limit || config.setting.limit)
    offset = parseInt(offset || 0)

    var result = await eventLogRepository.getAllLogs(
        limit,
        offset,
        sort || 'asc',
        sortBy || 'id',
        null,
        null,
        null,
        [logType.PAYMENT, logType.RECEIPT],
        id,
        req.httpContext
    )

    var count = await eventLogRepository.getLogCount(
        null,
        null,
        null,
        [logType.PAYMENT, logType.RECEIPT],
        id,
        req.httpContext
    )

    var pagination = generatePagination(offset, limit, count)

    var response = {
        result: result,
        pagination: pagination
    }

    return response
}

async function getPayment(req, res) {
    var id = req.params.paymentId;

    var creditPayment = await creditPaymentRepository.getPayment(id, req.httpContext,)

    if (creditPayment.isDeleted)
        throw Error(errorMessage.PAYMENT_DELETED)

    return creditPayment
}

async function createPayment(req, res) {
    var body = req.body;

    var creditPayment = new CreditPayment(body);

    creditPayment.receiptUrl = null
    creditPayment.status = null

    if (_.isEmpty(creditPayment.customerId))
        throw Error(errorMessage.CUSTOMER_ID_REQUIRED)

    var isExist = await creditUserRepository.isUserExist(creditPayment.customerId, req.httpContext,)

    if (!isExist)
        throw Error(errorMessage.CUSTOMER_NOT_FOUND)

    if (creditPayment.amount === null || creditPayment.amount === undefined)
        throw Error(errorMessage.AMOUNT_REQURIED)

    if (typeof creditPayment.amount != "number")
        throw Error(errorMessage.INVALID_AMOUNT)

    if (_.isEmpty(creditPayment.referenceNumber))
        throw Error(errorMessage.REFERENCE_NUMBER_REQUIRED)

    var isPaymentExist = await creditPaymentRepository.isPaymentExist(creditPayment.referenceNumber, req.httpContext)

    if (isPaymentExist)
        throw Error(errorMessage.REFERENCE_NUMBER_EXIST)

    var result = await creditPaymentRepository.createPayment(creditPayment, req.httpContext,);

    await utils.logEvent(req, actionType.CREATE, logType.PAYMENT, null, creditPayment, result.id)

    return result
}

async function updatePayment(req, res) {
    var id = req.params.paymentId

    var body = req.body;

    var creditPayment = await creditPaymentRepository.getPayment(id, req.httpContext,)

    if (creditPayment.isDeleted)
        throw Error(errorMessage.PAYMENT_DELETED)

    if (creditPayment.status == paymentStatus.APPROVED)
        throw Error(errorMessage.CANNOT_UPDATE_APPROVED_PAYMENT)

    var updateCreditPayment = new CreditPayment(body);
    delete updateCreditPayment.contra

    if (!_.isEmpty(updateCreditPayment.status)) {
        if (_.isEmpty(paymentStatus[body.status]))
            throw Error(errorMessage.INVALID_STATUS)

        if (_.isEmpty(creditPayment.receiptUrl))
            throw Error(errorMessage.CANNOT_UPDATE_PAYMENT_STATUS)

        if (updateCreditPayment.status == paymentStatus.APPROVED)
            updateCreditPayment.contra = updateCreditPayment.amount || creditPayment.amount
    }

    if (updateCreditPayment.amount !== null &&
        updateCreditPayment.amount !== undefined &&
        typeof updateCreditPayment.amount != "number")
        throw Error(errorMessage.INVALID_AMOUNT)

    delete updateCreditPayment.customerId
    delete updateCreditPayment.receiptUrl
    delete updateCreditPayment.referenceNumber

    if (!_.isEmpty(req.httpContext.identity.id_employee) &&
        !_.isEmpty(updateCreditPayment.status) &&
        updateCreditPayment.status != paymentStatus.PENDING) {

        await fxEmailService.sendPaymentUpdatedEmail(
            creditPayment.customerId,
            updateCreditPayment.status,
            updateCreditPayment.amount || creditPayment.amount,
            creditPayment.referenceNumber,
            updateCreditPayment.remarks || creditPayment.remarks,
            creditPayment.receiptUrl,
            req.httpContext,
        )
    }

    if (!_.isEmpty(updateCreditPayment.status) &&
        updateCreditPayment.status == paymentStatus.APPROVED) {

        var user = await creditUserRepository.getUser(creditPayment.customerId, req.httpContext)
        var updatedTotalContra = user.totalContra + updateCreditPayment.contra
        updatedTotalContra = round(updatedTotalContra, config.setting.decimal)
        await creditUserRepository.updateUser(creditPayment.customerId, { totalContra: updatedTotalContra }, req.httpContext)
    }

    var result = await creditPaymentRepository.updatePayment(id, updateCreditPayment, req.httpContext,)

    await utils.logEvent(req, actionType.UPDATE, logType.PAYMENT, creditPayment, updateCreditPayment, id)

    return result
}

async function uploadReceipt(req, res) {

    var file = req.files.file

    if (file === undefined)
        throw Error(errorMessage.FILE_REQUIRED)

    var fileExtension = mime.extension(file.mimetype)

    var paymentId = req.params.paymentId

    var creditPayment = await creditPaymentRepository.getPayment(paymentId, req.httpContext,)

    if (creditPayment.isDeleted)
        throw Error(errorMessage.PAYMENT_DELETED)

    var updateCreditPayment = {}

    var fileId = new ObjectId()
    var filename = "uploads/receipts/" + fileId.toHexString() + "." + fileExtension
    utils.ensureDirectoryExistence(filename)
    await file.mv(filename)

    updateCreditPayment.receiptUrl = req.protocol + '://' + req.headers.host + '/uploads/receipts/' + fileId.toString() + '.' + fileExtension
    updateCreditPayment.status = paymentStatus.PENDING

    if (!_.isEmpty(req.httpContext.identity.id_customer)) {
        var firstname = req.httpContext.identity.firstname
        var lastname = req.httpContext.identity.lastname
        var buyerName = !_.isEmpty(firstname) && !_.isEmpty(lastname) ? (firstname + ' ' + lastname) : null

        await fxEmailService.sendPaymentUploadedEmail(
            buyerName,
            creditPayment.amount,
            creditPayment.referenceNumber,
            updateCreditPayment.receiptUrl,
            req.httpContext,
        )
    }

    var result = await creditPaymentRepository.updatePayment(paymentId, updateCreditPayment, req.httpContext,)

    await utils.logEvent(req, actionType.UPLOAD, logType.RECEIPT, creditPayment, updateCreditPayment, paymentId)

    return result
}

async function uploadReceiptBase64(req, res) {
    var { base64, mimeType } = req.body

    if (_.isEmpty(base64))
        throw Error(errorMessage.BASE64_IS_REQUIRED)
    if (_.isEmpty(mimeType))
        throw Error(errorMessage.MIMETYPE_IS_REQUIRED)

    var fileExtension = mime.extension(mimeType)

    if (!fileExtension)
        throw Error(errorMessage.INVALID_MIMETYPE)

    var paymentId = req.params.paymentId

    var creditPayment = await creditPaymentRepository.getPayment(paymentId, req.httpContext,)

    if (creditPayment.isDeleted)
        throw Error(errorMessage.PAYMENT_DELETED)

    var updateCreditPayment = {}

    var fileId = new ObjectId()
    var filename = "uploads/receipts/" + fileId.toString() + "." + fileExtension
    utils.ensureDirectoryExistence(filename)
    fs.writeFileSync(filename, base64, 'base64')

    updateCreditPayment.receiptUrl = req.protocol + '://' + req.headers.host + '/uploads/receipts/' + fileId.toString() + '.' + fileExtension
    updateCreditPayment.status = paymentStatus.PENDING

    if (!_.isEmpty(req.httpContext.identity.id_customer)) {
        var firstname = req.httpContext.identity.firstname
        var lastname = req.httpContext.identity.lastname
        var buyerName = !_.isEmpty(firstname) && !_.isEmpty(lastname) ? (firstname + ' ' + lastname) : null

        await fxEmailService.sendPaymentUploadedEmail(
            buyerName,
            creditPayment.amount,
            creditPayment.referenceNumber,
            updateCreditPayment.receiptUrl,
            req.httpContext,
        )
    }

    var result = await creditPaymentRepository.updatePayment(paymentId, updateCreditPayment, req.httpContext,)

    await utils.logEvent(req, actionType.UPLOAD, logType.RECEIPT, creditPayment, updateCreditPayment, paymentId)

    return result
}

async function deleteReceipt(req, res) {
    var paymentId = req.params.paymentId

    var creditPayment = await creditPaymentRepository.getPayment(paymentId, req.httpContext,)

    if (creditPayment.isDeleted)
        throw Error(errorMessage.PAYMENT_DELETED)

    if (creditPayment.status == paymentStatus.APPROVED)
        throw Error(errorMessage.CANNOT_DELETE_RECEIPT)

    if (_.isEmpty(creditPayment.receiptUrl))
        throw Error(errorMessage.ALREADY_DELETED)

    var result = await creditPaymentRepository.deleteReceipt(paymentId, req.httpContext,)

    await utils.logEvent(req, actionType.DELETE, logType.RECEIPT, creditPayment, { receiptUrl: null }, paymentId)

    return result
}

async function deletePayment(req, res) {
    var paymentId = req.params.paymentId

    var creditPayment = await creditPaymentRepository.getPayment(paymentId, req.httpContext,)

    if (creditPayment.isDeleted)
        throw Error(errorMessage.ALREADY_DELETED)

    if (creditPayment.status == paymentStatus.APPROVED)
        throw Error(errorMessage.CANNOT_DELETE_PAYMENT)

    var result = await creditPaymentRepository.deletePayment(paymentId, req.httpContext,)

    await utils.logEvent(req, actionType.DELETE, logType.PAYMENT, creditPayment, { isDeleted: true }, paymentId)

    return result
}

module.exports = {
    getPaymentList,
    getPayment,
    createPayment,
    updatePayment,
    uploadReceipt,
    uploadReceiptBase64,
    deleteReceipt,
    deletePayment,
    getPaymentHistory,
}