var config = require('./../config');
var _ = require('lodash');
var uuid = require("uuid")
var utils = require("../utils/utils")
var creditUserRepository = require('../repository/creditUserRepository');
var eventLogRepository = require("../repository/eventLogRepository")
var CreditUser = require("../domain/creditUser");
var userStatus = require('../utils/creditUserStatus')
var errorMessage = require("../utils/errorMessage")
var logType = require("../utils/eventLogType")
var actionType = require("../utils/eventLogActionType")

var generatePagination = require('../model/pagination')

async function getUserList(req, res) {
    var {
        limit,
        offset,
        sort,
        sortBy,
        status,
        customerIds,
        customerId,
        toCreditLimit,
        fromCreditLimit,
        toCreditUsed,
        fromCreditUsed,
        toTotalCreditUsed,
        fromTotalCreditUsed,
        toCreatedDate,
        fromCreatedDate,
        toTotalContra,
        fromTotalContra,
    } = req.query

    limit = parseInt(limit || config.setting.limit)
    offset = parseInt(offset || 0)

    var result = await creditUserRepository.getAllUsers(
        limit,
        offset,
        sort || 'asc',
        sortBy || 'id',
        status,
        customerIds,
        customerId,
        parseFloat(toCreditLimit),
        parseFloat(fromCreditLimit),
        parseFloat(toCreditUsed),
        parseFloat(fromCreditUsed),
        parseFloat(toTotalCreditUsed),
        parseFloat(fromTotalCreditUsed),
        toCreatedDate,
        fromCreatedDate,
        parseFloat(toTotalContra),
        parseFloat(fromTotalContra),
        req.httpContext,
    );

    var count = await creditUserRepository.getUsersCount(
        status,
        customerIds,
        customerId,
        parseFloat(toCreditLimit),
        parseFloat(fromCreditLimit),
        parseFloat(toCreditUsed),
        parseFloat(fromCreditUsed),
        parseFloat(toTotalCreditUsed),
        parseFloat(fromTotalCreditUsed),
        toCreatedDate,
        fromCreatedDate,
        parseFloat(toTotalContra),
        parseFloat(fromTotalContra),
        req.httpContext,
    )

    var pagination = generatePagination(offset, limit, count)

    var response = {
        result: result,
        pagination: pagination
    }

    return response
}

async function getUser(req, res) {
    var id = req.params.customerId;

    return await creditUserRepository.getUser(id, req.httpContext,)
}

async function createUser(req, res) {
    var body = req.body;

    var creditUser = new CreditUser(body);
    creditUser.creditUsed = 0
    creditUser.totalCreditUsed = 0
    creditUser.totalContra = 0
    creditUser.status = userStatus[body.status] || userStatus.ENABLED
    creditUser.creditOrderReference = uuid.v4().substr(24, 12).toUpperCase()

    if (creditUser.creditLimit === null || creditUser.creditLimit === undefined)
        throw Error(errorMessage.CREDIT_LIMIT_REQUIRED)

    if (typeof creditUser.creditLimit != "number")
        throw Error(errorMessage.INVALID_CREDIT_LIMIT)

    if (_.isEmpty(creditUser.customerId))
        throw Error(errorMessage.CUSTOMER_ID_REQUIRED)

    var isExist = await creditUserRepository.isUserExist(creditUser.customerId, req.httpContext,)

    if (isExist)
        throw Error(errorMessage.CUSTOMER_ALREADY_EXIST)

    var result = await creditUserRepository.createUser(creditUser, req.httpContext,)

    await utils.logEvent(req, actionType.CREATE, logType.USER, null, creditUser, creditUser.customerId)

    return result
}

async function updateUser(req, res) {
    var id = req.params.customerId

    var body = req.body;

    var creditUser = await creditUserRepository.getUser(id, req.httpContext,)

    var updateCreditUser = new CreditUser(body);

    if (!_.isEmpty(updateCreditUser.status) && _.isEmpty(userStatus[body.status]))
        throw Error(errorMessage.INVALID_STATUS)

    if (updateCreditUser.creditLimit !== null && updateCreditUser.creditLimit !== undefined) {
        if (typeof updateCreditUser.creditLimit != "number")
            throw Error(errorMessage.INVALID_CREDIT_LIMIT)

        if (creditUser.creditUsed > updateCreditUser.creditLimit)
            throw Error(errorMessage.CREDIT_USED_EXCEEDED_CREDIT_LIMIT)
    }

    delete updateCreditUser.customerId
    delete updateCreditUser.creditUsed
    delete updateCreditUser.totalCreditUsed
    delete updateCreditUser.creditOrderReference
    delete updateCreditUser.totalContra

    var result = await creditUserRepository.updateUser(id, updateCreditUser, req.httpContext,)

    await utils.logEvent(req, actionType.UPDATE, logType.USER, creditUser, updateCreditUser, id)

    return result
}

async function getUserHistory(req, res) {
    var id = req.params.customerId

    var {
        limit,
        offset,
        sort,
        sortBy,
    } = req.query

    var isExist = await creditUserRepository.isUserExist(id, req.httpContext)

    if (!isExist)
        throw Error(errorMessage.CUSTOMER_NOT_FOUND)

    limit = parseInt(limit || config.setting.limit)
    offset = parseInt(offset || 0)

    var result = await eventLogRepository.getAllLogs(
        limit,
        offset,
        sort || 'asc',
        sortBy || 'id',
        null,
        null,
        logType.USER,
        null,
        id,
        req.httpContext
    )

    var count = await eventLogRepository.getLogCount(
        null,
        null,
        logType.USER,
        null,
        id,
        req.httpContext
    )

    var pagination = generatePagination(offset, limit, count)

    var response = {
        result: result,
        pagination: pagination
    }

    return response
}

module.exports = {
    getUserList,
    getUser,
    createUser,
    updateUser,
    getUserHistory,
}