var {
    insertOne,
    updateOne,
    patchOne
} = require("./mongoHelper");
var _ = require('lodash');
var db = require("../db")

var ObjectId = require("mongodb").ObjectID;
var CreditOrder = require("../domain/creditOrder");
var Seller = require("../domain/seller");
var Buyer = require("../domain/buyer");

var orderStatus = require("../utils/creditOrderStatus")
var orderInvoiceStatus = require("../utils/creditOrderInvoiceStatus")
var deliveryOrderStatus = require("../utils/deliveryOrderStatus")

var sortDir = require("../utils/sortDir")
var sortOption = require("../utils/creditOrderField")
var doSortOption = require("../utils/deliveryOrderField")

async function getAllOrders(
    limit,
    offset,
    sort,
    sortBy,
    orderId,
    orderIds,
    customerId,
    customerIds,
    userId,
    sellerId,
    sellerIds,
    status,
    statuses,
    invoiceStatus,
    invoiceStatuses,
    referenceNumber,
    toInvoiceAmount,
    fromInvoiceAmount,
    toOutstandingAmount,
    fromOutstandingAmount,
    toCreatedDate,
    fromCreatedDate,
    toAgingDate,
    fromAgingDate,
    toDeliverDate,
    fromDeliverDate,
    context,
) {
    var collection = db.instance().collection('creditOrder')

    var sortQuery = {};
    sortQuery[sortOption[sortBy] || '_id'] = sortDir[sort] || 1

    var whereQuery = []
    whereQuery.push({ clientId: context.client.id })
    if (!_.isEmpty(orderId)) {
        whereQuery.push({ 'orderId': orderId })
    }
    else if (!_.isEmpty(orderIds)) {
        var orderIdArray = JSON.parse(orderIds)
        whereQuery.push({ 'orderId': { $in: orderIdArray } })
    }
    if (!_.isEmpty(customerId)) {
        whereQuery.push({ 'buyer.customerId': customerId })
    }
    else if (!_.isEmpty(customerIds)) {
        var customerIdArray = JSON.parse(customerIds)
        whereQuery.push({ 'buyer.customerId': { $in: customerIdArray } })
    }
    if (!_.isEmpty(userId)) {
        whereQuery.push({ 'buyer.userId': userId })
    }
    if (!_.isEmpty(sellerId)) {
        whereQuery.push({ 'seller.sellerId': sellerId })
    }
    else if (!_.isEmpty(sellerIds)) {
        var sellerIdArray = JSON.parse(sellerIds)
        whereQuery.push({ 'seller.sellerId': { $in: sellerIdArray } })
    }
    if (!_.isEmpty(orderStatus[status])) {
        whereQuery.push({ 'status': orderStatus[status] })
    }
    else if (!_.isEmpty(statuses)) {
        var statusArray = JSON.parse(statuses)

        var isValid = statusArray.reduce(function (total, currentValue) {
            var valid = orderStatus[currentValue] != null
            return total && valid
        }, true)

        if (isValid)
            whereQuery.push({ 'status': { $in: statusArray } })
    }
    if (!_.isEmpty(orderInvoiceStatus[invoiceStatus])) {
        whereQuery.push({ 'invoiceStatus': orderInvoiceStatus[invoiceStatus] })
    }
    else if (!_.isEmpty(invoiceStatuses)) {
        var statusArray = JSON.parse(invoiceStatuses)

        var isValid = statusArray.reduce(function (total, currentValue) {
            var valid = orderInvoiceStatus[currentValue] != null
            return total && valid
        }, true)

        if (isValid)
            whereQuery.push({ 'invoiceStatus': { $in: statusArray } })
    }
    if (!_.isEmpty(referenceNumber)) {
        whereQuery.push({ 'referenceNumber': new RegExp(referenceNumber, 'i') })
    }
    if (!isNaN(toInvoiceAmount)) {
        whereQuery.push({ 'invoiceAmount': { $lte: toInvoiceAmount } })
    }
    if (!isNaN(fromInvoiceAmount)) {
        whereQuery.push({ 'invoiceAmount': { $gte: fromInvoiceAmount } })
    }
    if (!isNaN(toOutstandingAmount)) {
        whereQuery.push({ 'outstandingAmount': { $lte: toOutstandingAmount } })
    }
    if (!isNaN(fromOutstandingAmount)) {
        whereQuery.push({ 'outstandingAmount': { $gte: fromOutstandingAmount } })
    }
    if (!_.isEmpty(toCreatedDate))
        whereQuery.push({ 'createdDatetime': { $lte: new Date(toCreatedDate) } })

    if (!_.isEmpty(fromCreatedDate))
        whereQuery.push({ 'createdDatetime': { $gte: new Date(fromCreatedDate) } })

    if (!_.isEmpty(toAgingDate))
        whereQuery.push({ 'agingDatetime': { $lte: new Date(toAgingDate) } })

    if (!_.isEmpty(fromAgingDate))
        whereQuery.push({ 'agingDatetime': { $gte: new Date(fromAgingDate) } })

    if (!_.isEmpty(toDeliverDate))
        whereQuery.push({ 'deliverDatetime': { $lte: new Date(toDeliverDate) } })

    if (!_.isEmpty(fromDeliverDate))
        whereQuery.push({ 'deliverDatetime': { $gte: new Date(fromDeliverDate) } })

    var result = await collection
        .find(whereQuery.length === 0 ? {} : { $and: whereQuery })
        .skip(offset)
        .limit(limit)
        .sort(sortQuery)
        .collation({ locale: "en_US", numericOrdering: true })
        .toArray();

    return result.map(c => new CreditOrder(c));
}

async function getOrdersCount(
    orderId,
    orderIds,
    customerId,
    customerIds,
    userId,
    sellerId,
    sellerIds,
    status,
    statuses,
    invoiceStatus,
    invoiceStatuses,
    referenceNumber,
    toInvoiceAmount,
    fromInvoiceAmount,
    toOutstandingAmount,
    fromOutstandingAmount,
    toCreatedDate,
    fromCreatedDate,
    toAgingDate,
    fromAgingDate,
    toDeliverDate,
    fromDeliverDate,
    context,
) {
    var collection = db.instance().collection('creditOrder')

    var whereQuery = []
    whereQuery.push({ clientId: context.client.id })
    if (!_.isEmpty(orderId)) {
        whereQuery.push({ 'orderId': orderId })
    }
    else if (!_.isEmpty(orderIds)) {
        var orderIdArray = JSON.parse(orderIds)
        whereQuery.push({ 'orderId': { $in: orderIdArray } })
    }
    if (!_.isEmpty(customerId)) {
        whereQuery.push({ 'buyer.customerId': customerId })
    }
    else if (!_.isEmpty(customerIds)) {
        var customerIdArray = JSON.parse(customerIds)
        whereQuery.push({ 'buyer.customerId': { $in: customerIdArray } })
    }
    if (!_.isEmpty(userId)) {
        whereQuery.push({ 'buyer.userId': userId })
    }
    if (!_.isEmpty(sellerId)) {
        whereQuery.push({ 'seller.sellerId': sellerId })
    }
    else if (!_.isEmpty(sellerIds)) {
        var sellerIdArray = JSON.parse(sellerIds)
        whereQuery.push({ 'seller.sellerId': { $in: sellerIdArray } })
    }
    if (!_.isEmpty(orderStatus[status])) {
        whereQuery.push({ 'status': orderStatus[status] })
    }
    else if (!_.isEmpty(statuses)) {
        var statusArray = JSON.parse(statuses)

        var isValid = statusArray.reduce(function (total, currentValue) {
            var valid = orderStatus[currentValue] != null
            return total && valid
        }, true)

        if (isValid)
            whereQuery.push({ 'status': { $in: statusArray } })
    }
    if (!_.isEmpty(orderInvoiceStatus[invoiceStatus])) {
        whereQuery.push({ 'invoiceStatus': orderInvoiceStatus[invoiceStatus] })
    }
    else if (!_.isEmpty(invoiceStatuses)) {
        var statusArray = JSON.parse(invoiceStatuses)

        var isValid = statusArray.reduce(function (total, currentValue) {
            var valid = orderInvoiceStatus[currentValue] != null
            return total && valid
        }, true)

        if (isValid)
            whereQuery.push({ 'invoiceStatus': { $in: statusArray } })
    }
    if (!_.isEmpty(referenceNumber)) {
        whereQuery.push({ 'referenceNumber': new RegExp(referenceNumber, 'i') })
    }
    if (!isNaN(toInvoiceAmount)) {
        whereQuery.push({ 'invoiceAmount': { $lte: toInvoiceAmount } })
    }
    if (!isNaN(fromInvoiceAmount)) {
        whereQuery.push({ 'invoiceAmount': { $gte: fromInvoiceAmount } })
    }
    if (!isNaN(toOutstandingAmount)) {
        whereQuery.push({ 'outstandingAmount': { $lte: toOutstandingAmount } })
    }
    if (!isNaN(fromOutstandingAmount)) {
        whereQuery.push({ 'outstandingAmount': { $gte: fromOutstandingAmount } })
    }
    if (!_.isEmpty(toCreatedDate))
        whereQuery.push({ 'createdDatetime': { $lte: new Date(toCreatedDate) } })

    if (!_.isEmpty(fromCreatedDate))
        whereQuery.push({ 'createdDatetime': { $gte: new Date(fromCreatedDate) } })

    if (!_.isEmpty(toAgingDate))
        whereQuery.push({ 'agingDatetime': { $lte: new Date(toAgingDate) } })

    if (!_.isEmpty(fromAgingDate))
        whereQuery.push({ 'agingDatetime': { $gte: new Date(fromAgingDate) } })

    if (!_.isEmpty(toDeliverDate))
        whereQuery.push({ 'deliverDatetime': { $lte: new Date(toDeliverDate) } })

    if (!_.isEmpty(fromDeliverDate))
        whereQuery.push({ 'deliverDatetime': { $gte: new Date(fromDeliverDate) } })

    var result = await collection
        .count(whereQuery.length === 0 ? {} : { $and: whereQuery })

    return result
}

async function getOrder(id, context) {
    var collection = db.instance().collection('creditOrder')

    var result = await collection.findOne({ orderId: id, clientId: context.client.id });
    if (result)
        return new CreditOrder(result)
    else
        throw Error('Credit order not found')
}

async function isOrderExist(id, context) {
    var collection = db.instance().collection('creditOrder')

    var result = await collection.findOne({ orderId: id, clientId: context.client.id });
    if (result)
        return true
    else
        return false
}

async function createOrder(creditOrder, context) {
    var collection = db.instance().collection('creditOrder')

    delete creditOrder.id;

    var result = await insertOne(collection, creditOrder, context);
    var generatedData = customResultDatabase(result);
    return new CreditOrder(generatedData);
}

async function updateOrder(id, creditOrderObj, context) {
    var collection = db.instance().collection('creditOrder')

    var whereQuery = { orderId: id, clientId: context.client.id }

    var creditOrderPatch = {};
    if (_.isEmpty(creditOrderObj)) {
        throw Error("Must at least update 1 field")
    }
    if (!_.isEmpty(creditOrderObj.orderId)) {
        creditOrderPatch.orderId = creditOrderObj.orderId;
    }
    if (!_.isEmpty(creditOrderObj.buyer)) {
        creditOrderPatch.buyer = new Buyer(creditOrderObj.buyer)
    }
    if (!_.isEmpty(creditOrderObj.seller)) {
        creditOrderPatch.seller = new Seller(creditOrderObj.seller)
    }
    if (!_.isEmpty(creditOrderObj.remarks)) {
        creditOrderPatch.remarks = creditOrderObj.remarks;
    }
    if (creditOrderObj.invoiceAmount !== null && creditOrderObj.invoiceAmount !== undefined) {
        creditOrderPatch.invoiceAmount = creditOrderObj.invoiceAmount;
    }
    if (creditOrderObj.outstandingAmount !== null && creditOrderObj.outstandingAmount !== undefined) {
        creditOrderPatch.outstandingAmount = creditOrderObj.outstandingAmount;
    }
    if (creditOrderObj.totalOutstandingAmount !== null && creditOrderObj.totalOutstandingAmount !== undefined) {
        creditOrderPatch.totalOutstandingAmount = creditOrderObj.totalOutstandingAmount;
    }
    if (!_.isEmpty(creditOrderObj.status)) {
        creditOrderPatch.status = creditOrderObj.status;

        if (creditOrderObj.status == orderStatus.CLOSED ||
            creditOrderObj.status == orderStatus.CANCELLED ||
            creditOrderObj.status == orderStatus.REJECTED) {
            creditOrderPatch.agingDatetime = null
        }

    }
    if (!_.isEmpty(creditOrderObj.invoiceUrl)) {
        creditOrderPatch.invoiceUrl = creditOrderObj.invoiceUrl;
    }

    if (!_.isEmpty(creditOrderObj.invoiceStatus)) {
        creditOrderPatch.invoiceStatus = creditOrderObj.invoiceStatus
    }

    if (!_.isEmpty(creditOrderObj.referenceNumber)) {
        creditOrderPatch.referenceNumber = creditOrderObj.referenceNumber
    }

    if (!_.isEmpty(creditOrderObj.deliveryOrders)) {
        creditOrderPatch.deliveryOrders = creditOrderObj.deliveryOrders
    }

    if (creditOrderObj.agingDatetime != null) {
        creditOrderPatch.agingDatetime = creditOrderObj.agingDatetime
    }

    if (creditOrderObj.deliverDatetime != null) {
        creditOrderPatch.deliverDatetime = creditOrderObj.deliverDatetime
    }

    var {
        acknowledged,
        matchedCount,
        modifiedCount
    } = await patchOne(collection, whereQuery, creditOrderPatch, context)

    if (matchedCount !== 1)
        throw Error('No record found')
    return
}

async function deleteInvoice(id, context) {
    var collection = db.instance().collection('creditOrder')

    var whereQuery = { orderId: id, clientId: context.client.id }

    var {
        acknowledged,
        matchedCount,
        modifiedCount
    } = await patchOne(collection, whereQuery, { invoiceUrl: null, invoiceStatus: null }, context)

    if (matchedCount !== 1)
        throw Error('No record found')
    return
}

async function getDeliveryOrders(
    limit,
    offset,
    sort,
    sortBy,
    orderId,
    status,
    referenceNumber,
    context,
) {
    var collection = db.instance().collection('creditOrder')

    var sortQuery = {};
    sortQuery[doSortOption[sortBy] || 'id'] = sortDir[sort] || 1

    var aggregates = [
        { $unwind: '$deliveryOrders' },
        { $match: { clientId: context.client.id } },
        { $match: { referenceNumber: new RegExp(referenceNumber, "i") } },
        { $set: { "deliveryOrders.orderId": "$orderId" } },
        { $set: { "deliveryOrders.referenceNumber": "$referenceNumber" } },
        { $replaceRoot: { newRoot: "$deliveryOrders" } },
    ]

    if (!_.isEmpty(orderId)) {
        aggregates.push({ $match: { orderId: orderId } })
    }

    if (!_.isEmpty(deliveryOrderStatus[status])) {
        aggregates.push({ $match: { status: status } })
    }

    aggregates.push(
        { $set: { orderId: { $toInt: "$orderId" } } },
        { $sort: sortQuery },
        { $set: { orderId: { $toString: "$orderId" } } },
        { $skip: offset },
        { $limit: limit }
    )


    var result = await collection
        .aggregate(aggregates)
        .toArray();

    return result;
}

async function getDeliveryOrdersCount(
    orderId,
    status,
    referenceNumber,
    context,
) {
    var collection = db.instance().collection('creditOrder')

    var aggregates = [
        { $unwind: '$deliveryOrders' },
        { $match: { clientId: context.client.id } },
        { $match: { referenceNumber: new RegExp(referenceNumber, "i") } },
        { $set: { "deliveryOrders.orderId": "$orderId" } },
        { $set: { "deliveryOrders.referenceNumber": "$referenceNumber" } },
        { $replaceRoot: { newRoot: "$deliveryOrders" } },
    ]

    if (!_.isEmpty(orderId)) {
        aggregates.push({ $match: { orderId: orderId } })
    }

    if (!_.isEmpty(deliveryOrderStatus[status])) {
        aggregates.push({ $match: { status: status } })
    }

    aggregates.push({ $count: "count" })


    var result = await collection
        .aggregate(aggregates)
        .toArray();

    if (_.isEmpty(result))
        return 0;

    return result[0].count;
}

async function updateDeliveryOrder(id, deliveryOrders, context) {
    var collection = db.instance().collection('creditOrder')

    var whereQuery = { orderId: id, clientId: context.client.id }

    var {
        acknowledged,
        matchedCount,
        modifiedCount
    } = await collection.updateOne(whereQuery, {
        $set: { deliveryOrders: deliveryOrders, }
    })

    var {
        acknowledged,
        matchedCount,
        modifiedCount
    } = await patchOne(collection, whereQuery, { deliveryOrders: deliveryOrders }, context)

    if (matchedCount !== 1)
        throw Error('No record found')
    return
}

function customResultDatabase(result) {
    if (result && result.ops && result.ops.length == 1) {
        generatedResult = result.ops[0];
        return generatedResult;
    } else {
        return result.ops;
    }

}

module.exports = {
    getAllOrders,
    getOrder,
    createOrder,
    updateOrder,
    deleteInvoice,
    updateDeliveryOrder,
    isOrderExist,
    getOrdersCount,
    getDeliveryOrders,
    getDeliveryOrdersCount,
}