var {
    insertOne,
    updateOne,
    patchOne
} = require("./mongoHelper");
var _ = require('lodash');
var db = require("../db")

var ObjectId = require("mongodb").ObjectID;
var CreditOrderTransaction = require("../domain/creditOrderTransaction");

var transactionType = require("../utils/transactionType")
var transactionMovement = require("../utils/transactionMovement")

var sortDir = require("../utils/sortDir")
var sortOption = require("../utils/creditOrderTransactionField")

async function getAllTransactions(
    limit,
    offset,
    sort,
    sortBy,
    orderId,
    sign,
    paymentId,
    type,
    toChangeAmount,
    fromChangeAmount,
    context,
) {
    var collection = db.instance().collection('creditOrderTransaction')

    var sortQuery = {};
    sortQuery[sortOption[sortBy] || '_id'] = sortDir[sort] || 1

    var whereQuery = []
    whereQuery.push({ clientId: context.client.id })
    if (!_.isEmpty(orderId)) {
        whereQuery.push({ 'orderId': orderId })
    }
    if (!_.isEmpty(transactionMovement[sign])) {
        whereQuery.push({ 'sign': transactionMovement[sign] })
    }
    if (!_.isEmpty(paymentId)) {
        whereQuery.push({ 'paymentId': paymentId })
    }
    if (!_.isEmpty(transactionType[type])) {
        whereQuery.push({ 'type': transactionType[type] })
    }
    if (!isNaN(toChangeAmount)) {
        whereQuery.push({ 'changeAmount': { $lte: toChangeAmount } })
    }
    if (!isNaN(fromChangeAmount)) {
        whereQuery.push({ 'changeAmount': { $gte: fromChangeAmount } })
    }

    var result = await collection
        .find(whereQuery.length === 0 ? {} : { $and: whereQuery })
        .skip(offset)
        .limit(limit)
        .sort(sortQuery)
        .toArray();

    return result.map(c => new CreditOrderTransaction(c));
}

async function getTransactionsCount(
    orderId,
    sign,
    paymentId,
    type,
    toChangeAmount,
    fromChangeAmount,
    context,
) {
    var collection = db.instance().collection('creditOrderTransaction')

    var whereQuery = []
    whereQuery.push({ clientId: context.client.id })
    if (!_.isEmpty(orderId)) {
        whereQuery.push({ 'orderId': orderId })
    }
    if (!_.isEmpty(transactionMovement[sign])) {
        whereQuery.push({ 'sign': transactionMovement[sign] })
    }
    if (!_.isEmpty(paymentId)) {
        whereQuery.push({ 'paymentId': paymentId })
    }
    if (!_.isEmpty(transactionType[type])) {
        whereQuery.push({ 'type': transactionType[type] })
    }
    if (!isNaN(toChangeAmount)) {
        whereQuery.push({ 'changeAmount': { $lte: toChangeAmount } })
    }
    if (!isNaN(fromChangeAmount)) {
        whereQuery.push({ 'changeAmount': { $gte: fromChangeAmount } })
    }

    var result = await collection
        .count(whereQuery.length === 0 ? {} : { $and: whereQuery })

    return result
}

async function getTransaction(id, context) {
    var collection = db.instance().collection('creditOrderTransaction')

    var whereQuery = {
        "_id": new ObjectId(id),
        clientId: context.client.id
    }
    var result = await collection.findOne(whereQuery);
    if (result)
        return new CreditOrderTransaction(result)
    else
        throw Error('Transaction not found')
}

async function createTransaction(transaction, context) {
    var collection = db.instance().collection('creditOrderTransaction')

    delete transaction.id;

    var result = await insertOne(collection, transaction, context);
    var generatedData = customResultDatabase(result);
    return new CreditOrderTransaction(generatedData);
}

async function updateTransaction(id, transaction, context) {
    var collection = db.instance().collection('creditOrderTransaction')

    var whereQuery = {
        "_id": new ObjectId(id),
        clientId: context.client.id
    }
    if (_.isEmpty(transaction)) {
        throw Error('Must at least update 1 field')
    }
    var transactionPatch = {};
    if (!_.isEmpty(transaction.orderId)) {
        transactionPatch.orderId = transaction.orderId;
    }
    if (transaction.changeAmount !== null && transaction.changeAmount !== undefined) {
        transactionPatch.changeAmount = transaction.changeAmount;
    }
    if (transaction.previousAmount !== null && transaction.previousAmount !== undefined) {
        transactionPatch.previousAmount = transaction.previousAmount;
    }
    if (transaction.currentAmount !== null && transaction.currentAmount !== undefined) {
        transactionPatch.currentAmount = transaction.currentAmount;
    }
    if (!_.isEmpty(transaction.sign)) {
        transactionPatch.sign = transaction.sign;
    }
    if (!_.isEmpty(transaction.paymentId)) {
        transactionPatch.paymentId = transaction.paymentId;
    }
    if (!_.isEmpty(transaction.type)) {
        transactionPatch.type = transaction.type;
    }
    if (!_.isEmpty(transaction.metadata)) {
        transactionPatch.metadata = transaction.metadata;
    }

    var {
        acknowledged,
        matchedCount,
        modifiedCount
    } = await patchOne(collection, whereQuery, tarnsactionPatch, context)

    if (matchedCount !== 1)
        throw Error('No record found')
    return
}

function customResultDatabase(result) {
    if (result && result.ops && result.ops.length == 1) {
        generatedResult = result.ops[0];
        return generatedResult;
    } else {
        return result.ops;
    }

}

module.exports = {
    getAllTransactions,
    getTransaction,
    createTransaction,
    updateTransaction,
    getTransactionsCount,
}