var {
    insertOne,
    updateOne,
    patchOne
} = require("./mongoHelper");
var _ = require('lodash');
var db = require("../db")

var ObjectId = require("mongodb").ObjectID;
var CreditPayment = require("../domain/creditPayment");

var paymentStatus = require("../utils/creditPaymentStatus")

var sortDir = require("../utils/sortDir")
var sortOption = require("../utils/creditPaymentField")


async function getAllPayments(
    limit,
    offset,
    sort,
    sortBy,
    customerId,
    customerIds,
    status,
    referenceNumber,
    toAmount,
    fromAmount,
    toContra,
    fromContra,
    toCreatedDate,
    fromCreatedDate,
    context,
) {
    var collection = db.instance().collection('creditPayment')

    var sortQuery = {};
    sortQuery[sortOption[sortBy] || '_id'] = sortDir[sort] || 1

    var whereQuery = []
    whereQuery.push({ clientId: context.client.id })
    whereQuery.push({ 'isDeleted': { $ne: true } })
    if (!_.isEmpty(customerId)) {
        whereQuery.push({ 'customerId': customerId })
    }
    else if (!_.isEmpty(customerIds)) {
        var customerIdArray = JSON.parse(customerIds)
        whereQuery.push({ 'customerId': { $in: customerIdArray } })
    }
    if (!_.isEmpty(paymentStatus[status])) {
        whereQuery.push({ 'status': paymentStatus[status] })
    }
    if (!_.isEmpty(referenceNumber)) {
        whereQuery.push({ 'referenceNumber': new RegExp(referenceNumber, 'i') })
    }
    if (!isNaN(toAmount)) {
        whereQuery.push({ 'amount': { $lte: toAmount } })
    }
    if (!isNaN(fromAmount)) {
        whereQuery.push({ 'amount': { $gte: fromAmount } })
    }
    if (!isNaN(toContra)) {
        whereQuery.push({ 'contra': { $lte: toContra } })
    }
    if (!isNaN(fromContra)) {
        whereQuery.push({ 'contra': { $gte: fromContra } })
    }
    if (!_.isEmpty(toCreatedDate)) {
        whereQuery.push({ 'createdDatetime': { $lte: new Date(toCreatedDate) } })
    }
    if (!_.isEmpty(fromCreatedDate)) {
        whereQuery.push({ 'createdDatetime': { $gte: new Date(fromCreatedDate) } })
    }

    var result = await collection
        .find(whereQuery.length === 0 ? {} : { $and: whereQuery })
        .skip(offset)
        .limit(limit)
        .sort(sortQuery)
        .toArray();

    return result.map(c => new CreditPayment(c));
}

async function getPaymentsCount(
    customerId,
    customerIds,
    status,
    referenceNumber,
    toAmount,
    fromAmount,
    toContra,
    fromContra,
    toCreatedDate,
    fromCreatedDate,
    context,
) {
    var collection = db.instance().collection('creditPayment')

    var whereQuery = []
    whereQuery.push({ clientId: context.client.id })
    whereQuery.push({ 'isDeleted': { $ne: true } })
    if (!_.isEmpty(customerId)) {
        whereQuery.push({ 'customerId': customerId })
    }
    else if (!_.isEmpty(customerIds)) {
        var customerIdArray = JSON.parse(customerIds)
        whereQuery.push({ 'customerId': { $in: customerIdArray } })
    }
    if (!_.isEmpty(paymentStatus[status])) {
        whereQuery.push({ 'status': paymentStatus[status] })
    }
    if (!_.isEmpty(referenceNumber)) {
        whereQuery.push({ 'referenceNumber': new RegExp(referenceNumber, 'i') })
    }
    if (!isNaN(toAmount)) {
        whereQuery.push({ 'amount': { $lte: toAmount } })
    }
    if (!isNaN(fromAmount)) {
        whereQuery.push({ 'amount': { $gte: fromAmount } })
    }
    if (!isNaN(toContra)) {
        whereQuery.push({ 'contra': { $lte: toContra } })
    }
    if (!isNaN(fromContra)) {
        whereQuery.push({ 'contra': { $gte: fromContra } })
    }
    if (!_.isEmpty(toCreatedDate)) {
        whereQuery.push({ 'createdDatetime': { $lte: new Date(toCreatedDate) } })
    }
    if (!_.isEmpty(fromCreatedDate)) {
        whereQuery.push({ 'createdDatetime': { $gte: new Date(fromCreatedDate) } })
    }

    var result = await collection
        .count(whereQuery.length === 0 ? {} : { $and: whereQuery })

    return result
}

async function getPayment(id, context) {
    var collection = db.instance().collection('creditPayment')

    var whereQuery = {
        "_id": new ObjectId(id),
        clientId: context.client.id
    }
    var result = await collection.findOne(whereQuery);
    if (result)
        return new CreditPayment(result)
    else
        throw Error('Payment not found')
}

async function isPaymentExist(referenceNumber, context) {
    var collection = db.instance().collection('creditPayment')

    var whereQuery = { referenceNumber: referenceNumber, clientId: context.client.id }

    var result = await collection.findOne(whereQuery);
    if (result)
        return true
    else
        return false
}

async function createPayment(creditPayment, context) {
    var collection = db.instance().collection('creditPayment')

    delete creditPayment.id;
    creditPayment.isDeleted = false
    creditPayment.contra = 0

    var result = await insertOne(collection, creditPayment, context);
    var generatedData = customResultDatabase(result);
    return new CreditPayment(generatedData);
}

async function updatePayment(id, creditPayment, context) {
    var collection = db.instance().collection('creditPayment')

    var whereQuery = {
        "_id": new ObjectId(id),
        clientId: context.client.id
    }

    var creditPaymentPatch = {};
    if (_.isEmpty(creditPayment)) {
        return [new Error("No field updating"), false];
    }
    if (!_.isEmpty(creditPayment.customerId)) {
        creditPaymentPatch.customerId = creditPayment.customerId;
    }
    if (creditPayment.amount !== null && creditPayment.amount !== undefined) {
        creditPaymentPatch.amount = creditPayment.amount
    }
    if (creditPayment.contra !== null && creditPayment.contra !== undefined) {
        creditPaymentPatch.contra = creditPayment.contra
    }
    if (!_.isEmpty(creditPayment.receiptUrl)) {
        creditPaymentPatch.receiptUrl = creditPayment.receiptUrl;
    }
    if (!_.isEmpty(creditPayment.status)) {
        creditPaymentPatch.status = creditPayment.status;
    }
    if (!_.isEmpty(creditPayment.remarks)) {
        creditPaymentPatch.remarks = creditPayment.remarks;
    }
    if (!_.isEmpty(creditPayment.referenceNumber)) {
        creditPaymentPatch.referenceNumber = creditPayment.referenceNumber
    }

    var {
        acknowledged,
        matchedCount,
        modifiedCount
    } = await patchOne(collection, whereQuery, creditPaymentPatch, context)

    if (matchedCount !== 1)
        throw Error('No record found')
    return
}

async function deletePayment(id, context) {
    var collection = db.instance().collection('creditPayment')

    var whereQuery = {
        "_id": new ObjectId(id),
        clientId: context.client.id
    }

    var {
        acknowledged,
        matchedCount,
        modifiedCount
    } = await patchOne(collection, whereQuery, { isDeleted: true }, context)

    if (matchedCount !== 1)
        throw Error('No record found')
    return
}

async function deleteReceipt(id, context) {
    var collection = db.instance().collection('creditPayment')

    var whereQuery = {
        "_id": new ObjectId(id),
        clientId: context.client.id
    }

    var {
        acknowledged,
        matchedCount,
        modifiedCount
    } = await patchOne(collection, whereQuery, { receiptUrl: null }, context)

    if (matchedCount !== 1)
        throw Error('No record found')
    return
}

function customResultDatabase(result) {
    if (result && result.ops && result.ops.length == 1) {
        generatedResult = result.ops[0];
        return generatedResult;
    } else {
        return result.ops;
    }

}

module.exports = {
    getAllPayments,
    getPayment,
    createPayment,
    updatePayment,
    deleteReceipt,
    getPaymentsCount,
    deletePayment,
    isPaymentExist,
}